# SCP Integration Tests (Vitest)

Integration and E2E tests for the SCP API using [Vitest](https://vitest.dev/).

## Overview

This test suite validates the SCP API endpoints using the generated TypeScript SDK and client samples. Tests run against a live API server (local or deployed).

## Architecture

```
integration-tests/
├── package.json          # Vitest dependencies
├── vitest.config.ts      # Vitest configuration
├── vitest.setup.ts       # Test environment setup
├── tsconfig.json         # TypeScript configuration
└── tests/
    ├── tenants.test.ts   # Tenant API integration tests
    ├── roles.test.ts     # Role API integration tests
    └── ...               # More resource tests
```

## Quick Start

```bash
# Install dependencies
cd integration-tests && npm install

# Run all tests
npm test

# Watch mode (re-run on file changes)
npm run test:watch

# UI mode (visual test runner)
npm run test:ui

# Generate coverage report
npm run test:coverage
```

## Using Makefile

From the project root:

```bash
# Run integration tests
make test-integration

# Watch mode
make test-integration-watch

# UI mode
make test-integration-ui

# With coverage
make test-integration-coverage
```

## Environment Variables

Tests use environment variables from the root `.env` file:

- `BASE_URL` - API base URL (default: http://localhost:4000)
- `ADMIN_EMAIL` - Admin email for authentication
- `ADMIN_PASSWORD` - Admin password
- `TENANT_NAME` - Tenant name for authentication

## Writing Tests

### Example: Testing a Resource

```typescript
import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createResource } from '@scp/client-samples/resources/create';
import { getResource } from '@scp/client-samples/resources/show';

describe('Resource API Integration Tests', () => {
  beforeAll(async () => {
    // Authenticate once before all tests
    const response = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL,
        password: process.env.ADMIN_PASSWORD,
        tenant_name: process.env.TENANT_NAME,
      },
    });

    OpenAPI.TOKEN = response.token;
  });

  it('should create a resource', async () => {
    const resource = await createResource();

    expect(resource).toBeDefined();
    expect(resource.id).toBeTruthy();
  });

  it('should get a resource', async () => {
    const resource = await getResource('some-id');

    expect(resource).toBeDefined();
    expect(resource.id).toBe('some-id');
  });
});
```

## Integration with Client Samples

Tests import functions from `@scp/client-samples`:

```typescript
import { createTenant } from '@scp/client-samples/tenants/create';
import { listTenants } from '@scp/client-samples/tenants/list';
```

This ensures:
- ✅ Code samples in docs are validated
- ✅ SDK works correctly
- ✅ API behaves as expected
- ✅ Single source of truth for API usage examples

## vs K6 Load Tests

| Aspect | Vitest Integration | K6 Load Tests |
|--------|-------------------|---------------|
| **Purpose** | API correctness | Performance/load |
| **Speed** | Fast | Slower (high VUs) |
| **SDK** | Uses generated SDK | Uses ApiClient wrapper |
| **When to Run** | On every commit | Before releases |
| **Environment** | Node.js | K6 runtime |

## CI/CD Integration

Add to your CI pipeline:

```yaml
# .github/workflows/test.yml
jobs:
  integration-tests:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - name: Start API server
        run: mix phx.server &
      - name: Run integration tests
        run: make test-integration
```

## Troubleshooting

### "Connection refused"
Ensure the API server is running:
```bash
mix phx.server
```

### "Authentication failed"
Check your `.env` file has correct credentials:
```bash
cat .env | grep -E "ADMIN_EMAIL|ADMIN_PASSWORD|TENANT_NAME"
```

### "Module not found"
Regenerate the SDK:
```bash
make generate-k6-sdk
```

## Learn More

- [Vitest Documentation](https://vitest.dev/)
- [SCP API Documentation](../api-docs/)
- [Client Samples](../client_samples/typescript/)
