import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { listAuditLogs } from '@scp/client-samples/audit_logs/list';
import { getAuditLog } from '@scp/client-samples/audit_logs/show';

describe('AuditLog API Integration Tests', () => {
  let testTenantId: string;
  let testUserId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;
    testTenantId = authResponse.user.tenant_id;
    testUserId = authResponse.user.id;
  });

  it('should list audit logs (200)', async () => {
    const response = await listAuditLogs();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThanOrEqual(0);
  });

  it('should handle empty audit log list', async () => {
    const response = await listAuditLogs();

    expect(response).toBeDefined();
    expect(response.meta).toBeDefined();
  });

  // Note: We cannot test 'get single audit log' without creating one first,
  // and audit logs are immutable (created by system only).
  // If there are existing audit logs, we could test show functionality,
  // but we'll skip this test for now since we can't guarantee data exists.

  // Error case tests
  it('should fail to get non-existent audit log (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getAuditLog(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
