import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createBooking } from '@scp/client-samples/bookings/create';
import { listBookings } from '@scp/client-samples/bookings/list';
import { getBooking } from '@scp/client-samples/bookings/show';
import { updateBooking } from '@scp/client-samples/bookings/update';
import { deleteBooking } from '@scp/client-samples/bookings/delete';

describe('Booking API Integration Tests', () => {
  let bookingId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;
  });

  it('should create a booking (201)', async () => {
    const booking = await createBooking(testTenantId, testMerchantId, testEndCustomerId);

    expect(booking).toBeDefined();
    expect(booking.id).toBeTruthy();
    expect(booking.status).toBe('pending');

    bookingId = booking.id;
  });

  it('should list bookings (200)', async () => {
    const response = await listBookings();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundBooking = response.data!.find((b) => b.id === bookingId);
    expect(foundBooking).toBeDefined();
  });

  it('should get a single booking (200)', async () => {
    const booking = await getBooking(bookingId);

    expect(booking).toBeDefined();
    expect(booking.id).toBe(bookingId);
    expect(booking.status).toBe('pending');
  });

  it('should update a booking (200)', async () => {
    const updated = await updateBooking(
      bookingId,
      testTenantId,
      testMerchantId,
      testEndCustomerId
    );

    expect(updated).toBeDefined();
    expect(updated.id).toBe(bookingId);
    expect(updated.status).toBe('confirmed');
  });

  it('should delete a booking (200)', async () => {
    await deleteBooking(bookingId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getBooking(bookingId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create booking with missing required fields (422)', async () => {
    const { BookingService } = await import('@scp/sdk');

    try {
      await BookingService.bookingCreate({
        requestBody: {
          // Missing required fields
          status: 'pending',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(422);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to create booking with invalid status enum (422)', async () => {
    const { BookingService } = await import('@scp/sdk');

    try {
      await BookingService.bookingCreate({
        requestBody: {
          status: 'invalid_status',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(422);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to get non-existent booking (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getBooking(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to update non-existent booking (404 or 422)', async () => {
    const { BookingService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await BookingService.bookingUpdate({
        id: fakeId,
        requestBody: {
          status: 'confirmed',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent booking (404)', async () => {
    const { BookingService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await BookingService.bookingDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        expect(error).toBeDefined();
      }
    }
  });
});
