import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomerSearch } from '@scp/client-samples/end_customer_searches/create';
import { listEndCustomerSearches } from '@scp/client-samples/end_customer_searches/list';
import { getEndCustomerSearch } from '@scp/client-samples/end_customer_searches/show';
import { updateEndCustomerSearch } from '@scp/client-samples/end_customer_searches/update';
import { deleteEndCustomerSearch } from '@scp/client-samples/end_customer_searches/delete';


describe('EndCustomerSearch API Integration Tests', () => {
  let end_customer_searchId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testUserId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;
    testUserId = authResponse.user?.id;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a end_customer_search (201)', async () => {
    const end_customer_search = await createEndCustomerSearch(testTenantId, testMerchantId, testUserId);

    expect(end_customer_search).toBeDefined();
    expect(end_customer_search.id).toBeTruthy();

    end_customer_searchId = end_customer_search.id;
  });

  it('should list endcustomersearches (200)', async () => {
    const response = await listEndCustomerSearches();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single end_customer_search (200)', async () => {
    const end_customer_search = await getEndCustomerSearch(end_customer_searchId);

    expect(end_customer_search).toBeDefined();
    expect(end_customer_search.id).toBe(end_customer_searchId);
  });

  it('should update a end_customer_search (200)', async () => {
    const updated = await updateEndCustomerSearch(end_customer_searchId, testTenantId, testMerchantId, testUserId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(end_customer_searchId);
  });

  it('should delete a end_customer_search (204)', async () => {
    await deleteEndCustomerSearch(end_customer_searchId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getEndCustomerSearch(end_customer_searchId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent end_customer_search (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getEndCustomerSearch(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent end_customer_search (404)', async () => {
    const { EndCustomerSearchService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await EndCustomerSearchService.endCustomerSearchDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
