import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { listEndCustomers } from '@scp/client-samples/endcustomers/list';
import { getEndCustomer } from '@scp/client-samples/endcustomers/show';
import { updateEndCustomer } from '@scp/client-samples/endcustomers/update';
import { deleteEndCustomer } from '@scp/client-samples/endcustomers/delete';

describe('EndCustomer API Integration Tests', () => {
  let endCustomerId: string;
  let testTenantId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create an end customer (201)', async () => {
    const customer = await createEndCustomer(testTenantId, testMerchantId);

    expect(customer).toBeDefined();
    expect(customer.id).toBeTruthy();
    expect(customer.first_name).toBeTruthy();
    expect(customer.last_name).toBeTruthy();
    expect(customer.email).toBeTruthy();

    endCustomerId = customer.id;
  });

  it('should list end customers (200)', async () => {
    const response = await listEndCustomers();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundCustomer = response.data!.find((c) => c.id === endCustomerId);
    expect(foundCustomer).toBeDefined();
  });

  it('should get a single end customer (200)', async () => {
    const customer = await getEndCustomer(endCustomerId);

    expect(customer).toBeDefined();
    expect(customer.id).toBe(endCustomerId);
    expect(customer.first_name).toBeTruthy();
    expect(customer.last_name).toBeTruthy();
  });

  it('should update an end customer (200)', async () => {
    const updated = await updateEndCustomer(endCustomerId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(endCustomerId);
    expect(updated.first_name).toBeTruthy();
    expect(updated.last_name).toBeTruthy();
  });

  it('should delete an end customer (200)', async () => {
    await deleteEndCustomer(endCustomerId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getEndCustomer(endCustomerId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create end customer with missing required fields (422)', async () => {
    const { EndCustomerService } = await import('@scp/sdk');

    try {
      await EndCustomerService.endCustomerCreate({
        requestBody: {
          // Missing required fields: first_name, last_name, tenant_id, merchant_id
          email: 'invalid@example.com',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent end customer (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getEndCustomer(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent end customer (404 or 422)', async () => {
    const { EndCustomerService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await EndCustomerService.endCustomerUpdate({
        id: fakeId,
        requestBody: {
          first_name: 'Updated',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent end customer (404)', async () => {
    const { EndCustomerService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await EndCustomerService.endCustomerDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
