import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createInvoice } from '@scp/client-samples/invoices/create';
import { listInvoices } from '@scp/client-samples/invoices/list';
import { getInvoice } from '@scp/client-samples/invoices/show';
import { updateInvoice } from '@scp/client-samples/invoices/update';
import { deleteInvoice } from '@scp/client-samples/invoices/delete';


describe('Invoice API Integration Tests', () => {
  let invoiceId: string;
  let testTenantId: string;
  let testMerchantId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create an invoice (201)', async () => {
    const invoice = await createInvoice(testTenantId, testMerchantId);

    expect(invoice).toBeDefined();
    expect(invoice.id).toBeTruthy();
    expect(invoice.status).toBe('draft');

    invoiceId = invoice.id;
  });

  it('should list invoices (200)', async () => {
    const response = await listInvoices();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single invoice (200)', async () => {
    const invoice = await getInvoice(invoiceId);

    expect(invoice).toBeDefined();
    expect(invoice.id).toBe(invoiceId);
  });

  it('should update an invoice (200)', async () => {
    const updated = await updateInvoice(invoiceId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(invoiceId);
  });

  it('should delete an invoice (204)', async () => {
    await deleteInvoice(invoiceId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getInvoice(invoiceId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent invoice (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getInvoice(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent invoice (404)', async () => {
    const { InvoiceService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await InvoiceService.invoiceDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
