import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createLocation } from '@scp/client-samples/locations/create';
import { listLocations } from '@scp/client-samples/locations/list';
import { getLocation } from '@scp/client-samples/locations/show';
import { updateLocation } from '@scp/client-samples/locations/update';
import { deleteLocation } from '@scp/client-samples/locations/delete';

describe('Location API Integration Tests', () => {
  let locationId: string;
  let testTenantId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a location (201)', async () => {
    const location = await createLocation(testTenantId, testMerchantId);

    expect(location).toBeDefined();
    expect(location.id).toBeTruthy();
    expect(location.name).toBeTruthy();
    expect(location.city).toBeTruthy();

    locationId = location.id;
  });

  it('should list locations (200)', async () => {
    const response = await listLocations();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundLocation = response.data!.find((l) => l.id === locationId);
    expect(foundLocation).toBeDefined();
  });

  it('should get a single location (200)', async () => {
    const location = await getLocation(locationId);

    expect(location).toBeDefined();
    expect(location.id).toBe(locationId);
    expect(location.name).toBeTruthy();
  });

  it('should update a location (200)', async () => {
    const updated = await updateLocation(locationId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(locationId);
    expect(updated.name).toBeTruthy();
  });

  it('should delete a location (200)', async () => {
    await deleteLocation(locationId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getLocation(locationId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create location with missing required fields (422)', async () => {
    const { LocationService } = await import('@scp/sdk');

    try {
      await LocationService.locationCreate({
        requestBody: {
          // Missing required fields: name, address_line1, city, etc.
          postal_code: '12345',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent location (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getLocation(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent location (404 or 422)', async () => {
    const { LocationService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await LocationService.locationUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent location (404)', async () => {
    const { LocationService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await LocationService.locationDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
