import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService, TenantService, TenantRequest } from '@scp/sdk';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { listMerchantOrgs } from '@scp/client-samples/merchant_orgs/list';
import { getMerchantOrg } from '@scp/client-samples/merchant_orgs/show';
import { updateMerchantOrg } from '@scp/client-samples/merchant_orgs/update';
import { deleteMerchantOrg } from '@scp/client-samples/merchant_orgs/delete';

describe('MerchantOrg API Integration Tests', () => {
  let merchantOrgId: string;
  let testTenantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'root@scp.local',
        password: process.env.ADMIN_PASSWORD || 'Password123!',
        tenant_name: process.env.TENANT_NAME || 'scp',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create a test tenant for merchant orgs
    const tenantResponse = await TenantService.tenantCreate({
      requestBody: {
        name: `Test Tenant ${Date.now()}`,
        type: TenantRequest.type.TEST,
      },
    }) as any;

    // API returns {data: {...}}, extract the actual tenant
    const tenant = tenantResponse.data || tenantResponse;
    testTenantId = tenant.id!;
  });

  it('should create a merchant org', async () => {
    const merchantOrg = await createMerchantOrg(testTenantId);

    expect(merchantOrg).toBeDefined();
    expect(merchantOrg.id).toBeTruthy();
    expect(merchantOrg.name).toContain('Acme Corp');
    expect(merchantOrg.tenant_id).toBe(testTenantId);

    merchantOrgId = merchantOrg.id;
  });

  it('should list merchant orgs', async () => {
    const response = await listMerchantOrgs();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single merchant org', async () => {
    const merchantOrg = await getMerchantOrg(merchantOrgId);

    expect(merchantOrg).toBeDefined();
    expect(merchantOrg.id).toBe(merchantOrgId);
    expect(merchantOrg.tenant_id).toBe(testTenantId);
  });

  it('should update a merchant org', async () => {
    const updated = await updateMerchantOrg(merchantOrgId, testTenantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(merchantOrgId);
    expect(updated.name).toContain('Updated Corp');
  });

  it('should delete a merchant org', async () => {
    await deleteMerchantOrg(merchantOrgId);

    // Verify deletion
    await expect(async () => {
      await getMerchantOrg(merchantOrgId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create merchant org with missing required fields (422)', async () => {
    const { MerchantOrgService } = await import('@scp/sdk');

    try {
      await MerchantOrgService.merchantOrgCreate({
        requestBody: {
          // Missing required fields: name, tenant_id
          status: 'active',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent merchant org (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getMerchantOrg(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent merchant org (404 or 422)', async () => {
    const { MerchantOrgService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await MerchantOrgService.merchantOrgUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
          tenant_id: testTenantId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent merchant org (404)', async () => {
    const { MerchantOrgService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await MerchantOrgService.merchantOrgDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
