import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { listMerchants } from '@scp/client-samples/merchants/list';
import { getMerchant } from '@scp/client-samples/merchants/show';
import { updateMerchant } from '@scp/client-samples/merchants/update';
import { deleteMerchant } from '@scp/client-samples/merchants/delete';

describe('Merchant API Integration Tests', () => {
  let merchantId: string;
  let testTenantId: string;
  let testMerchantOrgId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test tenant and merchant org using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    testMerchantOrgId = merchantOrg.id;
  });

  it('should create a merchant (201)', async () => {
    const merchant = await createMerchant(testTenantId, testMerchantOrgId);

    expect(merchant).toBeDefined();
    expect(merchant.id).toBeTruthy();
    expect(merchant.name).toBeTruthy();
    expect(merchant.public_name).toBeTruthy();
    expect(merchant.tenant_id).toBe(testTenantId);
    expect(merchant.merchant_org_id).toBe(testMerchantOrgId);

    merchantId = merchant.id;
  });

  it('should list merchants (200)', async () => {
    const response = await listMerchants();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundMerchant = response.data!.find((m) => m.id === merchantId);
    expect(foundMerchant).toBeDefined();
  });

  it('should get a single merchant (200)', async () => {
    const merchant = await getMerchant(merchantId);

    expect(merchant).toBeDefined();
    expect(merchant.id).toBe(merchantId);
    expect(merchant.name).toBeTruthy();
    expect(merchant.public_name).toBeTruthy();
  });

  it('should update a merchant (200)', async () => {
    const updated = await updateMerchant(merchantId, testTenantId, testMerchantOrgId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(merchantId);
    expect(updated.name).toBeTruthy();
    expect(updated.public_name).toBeTruthy();
  });

  it('should delete a merchant (200)', async () => {
    await deleteMerchant(merchantId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getMerchant(merchantId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create merchant with missing required fields (422)', async () => {
    const { MerchantService } = await import('@scp/sdk');

    try {
      await MerchantService.merchantCreate({
        requestBody: {
          // Missing required fields: name, public_name, tenant_id, merchant_org_id
          status: 'active',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent merchant (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getMerchant(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent merchant (404 or 422)', async () => {
    const { MerchantService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await MerchantService.merchantUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
          tenant_id: testTenantId,
          merchant_org_id: testMerchantOrgId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent merchant (404)', async () => {
    const { MerchantService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await MerchantService.merchantDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
