import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { v4 as uuidv4 } from 'uuid';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createPaymentAccount } from '@scp/client-samples/payment_accounts/create';
import { listPaymentAccounts } from '@scp/client-samples/payment_accounts/list';
import { getPaymentAccount } from '@scp/client-samples/payment_accounts/show';
import { updatePaymentAccount } from '@scp/client-samples/payment_accounts/update';
import { deletePaymentAccount } from '@scp/client-samples/payment_accounts/delete';

describe('PaymentAccount API Integration Tests', () => {
  let paymentAccountId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;
  let testPluginId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;

    // Generate a plugin_id (in real scenarios this would be an actual plugin)
    testPluginId = uuidv4();
  });

  it('should create a payment account (201)', async () => {
    const paymentAccount = await createPaymentAccount(
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testPluginId
    );

    expect(paymentAccount).toBeDefined();
    expect(paymentAccount.id).toBeTruthy();
    expect(paymentAccount.type).toBe('card');
    expect(paymentAccount.brand).toBe('visa');
    expect(paymentAccount.last_4).toBe('4242');

    paymentAccountId = paymentAccount.id;
  });

  it('should list payment accounts (200)', async () => {
    const response = await listPaymentAccounts();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single payment account (200)', async () => {
    const paymentAccount = await getPaymentAccount(paymentAccountId);

    expect(paymentAccount).toBeDefined();
    expect(paymentAccount.id).toBe(paymentAccountId);
    expect(paymentAccount.type).toBe('card');
  });

  it('should update a payment account (200)', async () => {
    const updated = await updatePaymentAccount(
      paymentAccountId,
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testPluginId
    );

    expect(updated).toBeDefined();
    expect(updated.id).toBe(paymentAccountId);
    expect(updated.brand).toBe('mastercard');
    expect(updated.last_4).toBe('5555');
    expect(updated.is_default).toBe(true);
  });

  it('should delete a payment account (204)', async () => {
    await deletePaymentAccount(paymentAccountId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getPaymentAccount(paymentAccountId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent payment account (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getPaymentAccount(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent payment account (404)', async () => {
    const { PaymentAccountService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await PaymentAccountService.paymentAccountDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
