import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createProvider } from '@scp/client-samples/providers/create';
import { listProviders } from '@scp/client-samples/providers/list';
import { getProvider } from '@scp/client-samples/providers/show';
import { updateProvider } from '@scp/client-samples/providers/update';
import { deleteProvider } from '@scp/client-samples/providers/delete';

describe('Provider API Integration Tests', () => {
  let providerId: string;
  let testTenantId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a provider (201)', async () => {
    const provider = await createProvider(testTenantId, testMerchantId);

    expect(provider).toBeDefined();
    expect(provider.id).toBeTruthy();
    expect(provider.name).toBeTruthy();
    expect(provider.headline).toBeTruthy();

    providerId = provider.id;
  });

  it('should list providers (200)', async () => {
    const response = await listProviders();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundProvider = response.data!.find((p) => p.id === providerId);
    expect(foundProvider).toBeDefined();
  });

  it('should get a single provider (200)', async () => {
    const provider = await getProvider(providerId);

    expect(provider).toBeDefined();
    expect(provider.id).toBe(providerId);
    expect(provider.name).toBeTruthy();
  });

  it('should update a provider (200)', async () => {
    const updated = await updateProvider(providerId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(providerId);
    expect(updated.name).toBeTruthy();
  });

  it('should delete a provider (200)', async () => {
    await deleteProvider(providerId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getProvider(providerId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create provider with missing required fields (422)', async () => {
    const { ProviderService } = await import('@scp/sdk');

    try {
      await ProviderService.providerCreate({
        requestBody: {
          // Missing required fields: name, tenant_id, merchant_id
          headline: 'Invalid provider',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent provider (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getProvider(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent provider (404 or 422)', async () => {
    const { ProviderService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await ProviderService.providerUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent provider (404)', async () => {
    const { ProviderService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await ProviderService.providerDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
