import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService, TenantService, TenantRequest } from '@scp/sdk';
import { createRole } from '@scp/client-samples/roles/create';
import { listRoles } from '@scp/client-samples/roles/list';
import { getRole } from '@scp/client-samples/roles/show';
import { updateRole } from '@scp/client-samples/roles/update';
import { deleteRole } from '@scp/client-samples/roles/delete';

describe('Role API Integration Tests', () => {
  let roleId: string;
  let testTenantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'root@scp.local',
        password: process.env.ADMIN_PASSWORD || 'Password123!',
        tenant_name: process.env.TENANT_NAME || 'scp',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create a test tenant for roles
    const tenantResponse = await TenantService.tenantCreate({
      requestBody: {
        name: `Test Tenant ${Date.now()}`,
        type: TenantRequest.type.TEST,
      },
    }) as any;

    // API returns {data: {...}}, extract the actual tenant
    const tenant = tenantResponse.data || tenantResponse;
    testTenantId = tenant.id!;
  });

  it('should create a role', async () => {
    const role = await createRole(testTenantId);

    expect(role).toBeDefined();
    expect(role.id).toBeTruthy();
    expect(role.name).toContain('test_role_');
    expect(role.tenant_id).toBe(testTenantId);

    roleId = role.id;
  });

  it('should list roles', async () => {
    const response = await listRoles();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single role', async () => {
    const role = await getRole(roleId);

    expect(role).toBeDefined();
    expect(role.id).toBe(roleId);
    expect(role.tenant_id).toBe(testTenantId);
  });

  it('should update a role', async () => {
    const updated = await updateRole(roleId, testTenantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(roleId);
    expect(updated.name).toContain('updated_role_');
  });

  it('should delete a role', async () => {
    await deleteRole(roleId);

    // Verify deletion
    await expect(async () => {
      await getRole(roleId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create role with missing required fields (422)', async () => {
    const { RoleService } = await import('@scp/sdk');

    try {
      await RoleService.roleCreate({
        requestBody: {
          // Missing required fields: name, tenant_id
          permissions: [],
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent role (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getRole(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent role (404 or 422)', async () => {
    const { RoleService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      const result = await RoleService.roleUpdate({
        id: fakeId,
        requestBody: {
          name: 'updated_name',
          tenant_id: testTenantId,
        } as any,
      });
      // If no error was thrown, check if result indicates failure
      if (result) {
        expect.fail('Should have thrown error or returned error status');
      }
    } catch (error: any) {
      // Accept error with status or any thrown error
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        // Any error is acceptable for non-existent resource
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent role (404)', async () => {
    const { RoleService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await RoleService.roleDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
