import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createService } from '@scp/client-samples/services/create';
import { listServices } from '@scp/client-samples/services/list';
import { getService } from '@scp/client-samples/services/show';
import { updateService } from '@scp/client-samples/services/update';
import { deleteService } from '@scp/client-samples/services/delete';

describe('Service API Integration Tests', () => {
  let serviceId: string;
  let testTenantId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a service (201)', async () => {
    const service = await createService(testTenantId, testMerchantId);

    expect(service).toBeDefined();
    expect(service.id).toBeTruthy();
    expect(service.name).toBeTruthy();
    expect(service.duration).toBeTruthy();
    expect(service.standard_rate_str).toBeTruthy();

    serviceId = service.id;
  });

  it('should list services (200)', async () => {
    const response = await listServices();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundService = response.data!.find((s) => s.id === serviceId);
    expect(foundService).toBeDefined();
  });

  it('should get a single service (200)', async () => {
    const service = await getService(serviceId);

    expect(service).toBeDefined();
    expect(service.id).toBe(serviceId);
    expect(service.name).toBeTruthy();
  });

  it('should update a service (200)', async () => {
    const updated = await updateService(serviceId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(serviceId);
    expect(updated.name).toBeTruthy();
  });

  it('should delete a service (200)', async () => {
    await deleteService(serviceId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getService(serviceId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create service with missing required fields (422)', async () => {
    const { ServiceService } = await import('@scp/sdk');

    try {
      await ServiceService.serviceCreate({
        requestBody: {
          // Missing required fields: name, tenant_id, merchant_id
          description: 'Invalid service',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to create service with invalid type enum (422)', async () => {
    const { ServiceService } = await import('@scp/sdk');

    try {
      await ServiceService.serviceCreate({
        requestBody: {
          name: 'Invalid Service',
          type: 'invalid_type_value',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent service (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getService(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent service (404 or 422)', async () => {
    const { ServiceService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await ServiceService.serviceUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      // Accept either 404 (not found) or 422 (validation error)
      expect([404, 422]).toContain(error.status);
    }
  });

  it('should fail to delete non-existent service (404)', async () => {
    const { ServiceService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await ServiceService.serviceDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
