import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createSubscription } from '@scp/client-samples/subscriptions/create';
import { listSubscriptions } from '@scp/client-samples/subscriptions/list';
import { getSubscription } from '@scp/client-samples/subscriptions/show';
import { updateSubscription } from '@scp/client-samples/subscriptions/update';
import { deleteSubscription } from '@scp/client-samples/subscriptions/delete';


describe('Subscription API Integration Tests', () => {
  let subscriptionId: string;
  let testTenantId: string;
  let testMerchantId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a subscription (201)', async () => {
    const subscription = await createSubscription(testTenantId, testMerchantId);

    expect(subscription).toBeDefined();
    expect(subscription.id).toBeTruthy();

    subscriptionId = subscription.id;
  });

  it('should list subscriptions (200)', async () => {
    const response = await listSubscriptions();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single subscription (200)', async () => {
    const subscription = await getSubscription(subscriptionId);

    expect(subscription).toBeDefined();
    expect(subscription.id).toBe(subscriptionId);
  });

  it('should update a subscription (200)', async () => {
    const updated = await updateSubscription(subscriptionId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(subscriptionId);
  });

  it('should delete a subscription (204)', async () => {
    await deleteSubscription(subscriptionId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getSubscription(subscriptionId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent subscription (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getSubscription(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent subscription (404)', async () => {
    const { SubscriptionService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await SubscriptionService.subscriptionDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
