import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService, TransactionJournalService } from '@scp/sdk';
import { listTransactionJournals } from '@scp/client-samples/transaction_journals/list';
import { getTransactionJournal } from '@scp/client-samples/transaction_journals/show';

describe('TransactionJournal API Integration Tests', () => {
  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;
  });

  // Note: TransactionJournal is read-only from the API.
  // Journals are created internally when transactions are processed.
  // These tests verify the read operations work correctly.

  it('should list transaction_journals (200)', async () => {
    const response = await listTransactionJournals();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  // Note: Testing show requires an existing journal which is created
  // internally. We test the error case for non-existent ID.
  it('should fail to get non-existent transaction_journal (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getTransactionJournal(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail with invalid UUID format (422)', async () => {
    try {
      await TransactionJournalService.transactionJournalGet({ id: 'invalid-uuid' });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
    }
  });
});
