import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createTransaction } from '@scp/client-samples/transactions/create';
import { listTransactions } from '@scp/client-samples/transactions/list';
import { getTransaction } from '@scp/client-samples/transactions/show';
import { updateTransaction } from '@scp/client-samples/transactions/update';
import { deleteTransaction } from '@scp/client-samples/transactions/delete';

// Dummy plugin ID for legacy field (FK constraint removed, plugin system eliminated)
const DUMMY_PLUGIN_ID = '00000000-0000-0000-0000-000000000001';

describe('Transaction API Integration Tests', () => {
  let transactionId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    // Create end customer for transaction tests
    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;
  });

  it('should create a transaction (201)', async () => {
    const transaction = await createTransaction(testTenantId, testMerchantId, testEndCustomerId, DUMMY_PLUGIN_ID);

    expect(transaction).toBeDefined();
    expect(transaction.id).toBeTruthy();

    transactionId = transaction.id;
  });

  it('should list transactions (200)', async () => {
    const response = await listTransactions();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single transaction (200)', async () => {
    const transaction = await getTransaction(transactionId);

    expect(transaction).toBeDefined();
    expect(transaction.id).toBe(transactionId);
  });

  it('should update a transaction (200)', async () => {
    const updated = await updateTransaction(transactionId, testTenantId, testMerchantId, testEndCustomerId, DUMMY_PLUGIN_ID);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(transactionId);
  });

  it('should delete a transaction (204)', async () => {
    await deleteTransaction(transactionId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getTransaction(transactionId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent transaction (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getTransaction(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent transaction (404)', async () => {
    const { TransactionService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await TransactionService.transactionDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
